import os
import subprocess
import shutil
import requests
from db import load_db, register_package
from utils import download_extract, version_satisfies, BUILD_DIR, PKG_DIR

TMP_RECIPE_DIR = "/tmp/fempkg"
os.makedirs(TMP_RECIPE_DIR, exist_ok=True)

RECIPE_CACHE_DIR = os.path.expanduser("/var/lib/fempkg/repo")
MANIFEST_CACHE_DIR = "/var/lib/fempkg/manifests"
os.makedirs(RECIPE_CACHE_DIR, exist_ok=True)
os.makedirs(MANIFEST_CACHE_DIR, exist_ok=True)

def safe_rmtree(path):
    """Remove path recursively or symlink target."""
    if os.path.islink(path):
        real_path = os.path.realpath(path)
        if os.path.exists(real_path):
            shutil.rmtree(real_path, ignore_errors=True)
        os.unlink(path)
    elif os.path.exists(path):
        shutil.rmtree(path, ignore_errors=True)

def _ensure_symlink(src: str, target: str):
    if os.path.islink(src):
        if os.readlink(src) == target:
            return
        else:
            os.unlink(src)
    elif os.path.exists(src):
        shutil.rmtree(src, ignore_errors=True)
    os.makedirs(target, exist_ok=True)
    os.symlink(target, src)

_ensure_symlink("/tmp/fempkg", "/var/tmp/fempkg")
_ensure_symlink("/tmp/fempkgbuild", "/var/tmp/fempkgbuild")

# --------------------------
# Fetch recipes & manifests
# --------------------------
def fetch_all_recipes(repo_url="https://rocketleaguechatp.duckdns.org/recipes"):
    index_url = f"{repo_url}/index.txt"
    print(f"Fetching recipe index from {index_url}")
    try:
        with requests.Session() as session:
            resp = session.get(index_url)
            resp.raise_for_status()
            entries = resp.text.splitlines()
            for entry in entries:
                if entry.endswith(".recipe.py"):
                    url = f"{repo_url}/{entry}"
                    dest_path = os.path.join(RECIPE_CACHE_DIR, entry)
                    print(f"Downloading recipe {url} -> {dest_path}")
                    r = session.get(url)
                    r.raise_for_status()
                    with open(dest_path, "wb") as f:
                        f.write(r.content)
    except Exception as e:
        print(f"Error fetching index or recipes: {e}")

def fetch_all_manifests(repo_url="https://rocketleaguechatp.duckdns.org/manifests"):
    index_url = f"{repo_url}/index.txt"
    print(f"Fetching manifest index from {index_url}")
    try:
        with requests.Session() as session:
            resp = session.get(index_url)
            resp.raise_for_status()
            entries = resp.text.splitlines()
            for entry in entries:
                if entry.endswith(".txt"):
                    url = f"{repo_url}/{entry}"
                    dest_path = os.path.join(MANIFEST_CACHE_DIR, entry)
                    print(f"Downloading manifest {url} -> {dest_path}")
                    r = session.get(url)
                    r.raise_for_status()
                    with open(dest_path, "wb") as f:
                        f.write(r.content)
    except Exception as e:
        print(f"Error fetching index or manifests: {e}")

def fetch_all(repo_url_recipes=None, repo_url_manifests=None):
    fetch_all_recipes(repo_url_recipes or "https://rocketleaguechatp.duckdns.org/recipes")
    fetch_all_manifests(repo_url_manifests or "https://rocketleaguechatp.duckdns.org/manifests")
    print("All recipes and manifests fetched successfully.")

# --------------------------
# Recipe / manifest helpers
# --------------------------
def fetch_recipe(pkgname):
    path = os.path.join(RECIPE_CACHE_DIR, f"{pkgname}.recipe.py")
    if not os.path.exists(path):
        raise FileNotFoundError(f"Recipe for {pkgname} not found in cache. Run `fempkg update` first.")
    return path

def fetch_manifest(pkgname, pkgver=None):
    path = os.path.join(MANIFEST_CACHE_DIR, f"{pkgname}.txt")
    if not os.path.exists(path):
        raise FileNotFoundError(f"Manifest for {pkgname} not found. Run `fempkg update` first.")
    if not pkgver:
        return path
    temp_path = os.path.join(MANIFEST_CACHE_DIR, f"{pkgname}-resolved.txt")
    with open(path) as f:
        content = f.read()
    content = content.replace("{pkgver}", pkgver)
    with open(temp_path, "w") as f:
        f.write(content)
    return temp_path

# --------------------------
# Build packages
# --------------------------
def build_package(recipe_file, repo_dir=None):
    _ensure_symlink("/tmp/fempkg", "/var/tmp/fempkg")
    _ensure_symlink("/tmp/fempkgbuild", "/var/tmp/fempkgbuild")
    _ensure_symlink("/tmp/fempkg", "/var/tmp/fempkg")
    _ensure_symlink("/tmp/fempkgbuild", "/var/tmp/fempkgbuild")
    db = load_db()
    recipe = {}
    with open(recipe_file, "r") as f:
        exec(f.read(), recipe)

    name, version = recipe["pkgname"], recipe["pkgver"]
    source_type = recipe.get("source_type")
    deps = recipe.get("deps", [])

    # Handle dependencies
    for dep_name in deps:
        dep_recipe = None
        if repo_dir:
            dep_recipe = os.path.join(repo_dir, f"{dep_name}.recipe.py")
            if not os.path.exists(dep_recipe):
                print(f"Warning: recipe for {dep_name} not found in {repo_dir}.")
                dep_recipe = None
        if not dep_recipe:
            dep_recipe = fetch_recipe(dep_name)
        dep_info = {}
        with open(dep_recipe, "r") as f:
            exec(f.read(), dep_info)
        dep_latest_ver = dep_info["pkgver"]
        installed_ver = db["installed"].get(dep_name)
        if installed_ver is None or not version_satisfies(installed_ver, dep_latest_ver):
            print(f"Installing/updating dependency {dep_name} "
                  f"(installed: {installed_ver}, latest: {dep_latest_ver})")
            build_package(dep_recipe, repo_dir)
        else:
            print(f"Dependency {dep_name} is up-to-date ({installed_ver}). Skipping.")

    # Skip if already installed
    if name in db["installed"] and version_satisfies(db["installed"][name], version):
        print(f"{name}-{version} already installed and up-to-date.")
        return

    # Use manifest
    manifest_path = fetch_manifest(name, pkgver=version)
    with open(manifest_path) as f:
        files = sorted(line.strip() for line in f if line.strip())
    print(f"Using manifest for {name} ({len(files)} files)")

    # Download/extract source
    download_extract(recipe["source"], source_type)
    for cmd in recipe.get("build", []):
        print(f"> {cmd}")
        subprocess.run(f". /etc/profile && mkdir -p /tmp/fempkg && {cmd}", shell=True, check=True)

    # Cleanup
    for cleanup_path in ["/tmp/fempkg", "/tmp/fempkgbuild"]:
        target = os.path.realpath(cleanup_path)
        if os.path.exists(target):
            shutil.rmtree(target, ignore_errors=True)
            os.makedirs(target, exist_ok=True)

    # Remove tarball
    basename = os.path.basename(recipe["source"])
    tarball_path = os.path.join(PKG_DIR, basename)
    if os.path.exists(tarball_path):
        os.remove(tarball_path)

    register_package(name, version, db=db)
