import os, shutil, ssl, urllib.request, subprocess, errno
from urllib.parse import urlparse

BUILD_DIR = "/tmp/fempkgbuild"
PKG_DIR = "/var/lib/fempkg/pkgs"

os.makedirs(PKG_DIR, exist_ok=True)
os.makedirs(BUILD_DIR, exist_ok=True)

def _ensure_dir(path):
    try: os.makedirs(path, exist_ok=True)
    except OSError as e:
        if e.errno != errno.EEXIST: raise

def version_satisfies(installed, latest):
    def parse(v):
        parts = v.replace("-", ".").split(".")
        parsed = []
        for p in parts:
            parsed.append(int(p) if p.isdigit() else p)
        return parsed
    return parse(installed) >= parse(latest)

def _install_insecure_opener_if_requested():
    if os.environ.get("FEMPKG_INSECURE") == "1":
        ctx = ssl.create_default_context()
        ctx.check_hostname = False
        ctx.verify_mode = ssl.CERT_NONE
        opener = urllib.request.build_opener(urllib.request.HTTPSHandler(context=ctx))
        urllib.request.install_opener(opener)

def _safe_basename_from_url(url):
    parsed = urlparse(url)
    name = os.path.basename(parsed.path)
    return name or "downloaded.file"

def download_to(url, dest):
    _install_insecure_opener_if_requested()
    os.makedirs(os.path.dirname(dest), exist_ok=True)
    urllib.request.urlretrieve(url, dest)

def _strip_archive_ext(name):
    for ext in [".tar.gz", ".tar.bz2", ".tar.xz", ".tgz", ".tar"]:
        if name.endswith(ext):
            return name[:-len(ext)]
    return os.path.splitext(name)[0]

def _clone_git_repo(url):
    repo_name = os.path.splitext(os.path.basename(url))[0]
    dest = os.path.join(BUILD_DIR, repo_name)
    if os.path.exists(dest):
        shutil.rmtree(dest)
    os.makedirs(BUILD_DIR, exist_ok=True)
    subprocess.check_call(["git", "clone", url, dest])
    return dest

def download_extract(url, source_type):
    _install_insecure_opener_if_requested()
    if source_type == "git":
        return _clone_git_repo(url)
    basename = _safe_basename_from_url(url)
    tarball_path = os.path.join(PKG_DIR, basename)
    if not os.path.exists(tarball_path):
        download_to(url, tarball_path)
    dest_dir = os.path.join(BUILD_DIR, _strip_archive_ext(basename))
    if os.path.exists(dest_dir):
        shutil.rmtree(dest_dir)
    os.makedirs(dest_dir)
    return dest_dir
